/*
Author: Mike Brauch
Date: 5/11/2014
Description: This project was built around being able to read data from a temperature sensor and to display this data to the user in a
useful manner relating to time. Using an HD44780 20x4 White Text on Blue Background LCD, you were able to start this application via a ethernet TCP socket
controlled by a Java application on a computer. Via this application, you are able to specify the run time, the interval in which it should take readings,
pausing/resuming the application, and stopping it. On the event of a stop button it will take what data it has gathered and calculate the final information.
Upon reaching the max run time, it will also calculate this information. 
Final Info Includes:
Average Change in Temp in both Celsius and Farenheit
Average Overall Temperature in both Celsius and Farenheit
Total Run Time

MBED Libraries Used: EthernetInterface, FatFileSystem, RTOS, MSCFileSystem, TextLCD
Concepts Used from Class: Sockets, Mailbox, Pthreads, Real-time tasks
*/
#include "mbed.h"
#include "MSCFileSystem.h"
#include "EthernetInterface.h"
#include "TextLCD.h"
#include "rtos.h"
 
#define SERVER_PORT   20
#define FSNAME "msc"

// Temperature Structure 
typedef struct {
  float    tempC;
  float    tempF;
  float    timeStamp;
  float    avgC; // average change in celsius per second
  float    avgF; //average change in farenheit per second
} Temperature;

/******* Defines all Analog/Digital devices connected to microcontroller*******/
MSCFileSystem myUSBPort(FSNAME);
DigitalOut led1(LED1);
DigitalOut led2(LED2);
DigitalOut led3(LED3);
DigitalOut led4(LED4);
TextLCD lcd(p10, p12, p15, p16, p29, p30, TextLCD::LCD20x4); // rs, e, d0-d3
AnalogIn tempSensor(p17); //Temeperature sensor
PwmOut Speaker(p21);

Semaphore SemFlag(2);
Mail<Temperature, 16> mail_box;  // allocatable queue - very similar to FIFO
Timer t; //timer
FILE *fp; //temperature logging File
bool cont = true, pause = false; //boolean for stopping and pausing application
int numOfReadings = 0; //total number of temperatures read
float totalC = 0, totalF = 0; //total temp read
int TimeToRun = 0, timeInterval = 0;  //how long the application runs given by JAVA application
TCPSocketConnection client; //client connecting to JAVA app
int pauseFlag = 5;

/********** This function takes the pointer to mail and shared buffer to calculate and store temperature from sensor *******/
void calcTemp(Temperature *mail, void const *args){
    Temperature* myTemp = (Temperature*)args;
    float holdC, holdF;
    //conversion to degrees C
        holdC = myTemp->tempC;
        holdF = myTemp->tempF;
        
        mail->tempC = ((tempSensor*3.3)-0.600)*100.0;
        myTemp->tempC = ((tempSensor*3.3)-0.600)*100.0;
        //convert to degrees F
        mail->tempF = (9.0*mail->tempC)/5.0 + 32.0;
        myTemp->tempF = (9.0*mail->tempC)/5.0 + 32.0;
        //storing time
        mail->timeStamp = t.read();
        myTemp->timeStamp = t.read();
        //computing average change in temperature
        mail->avgC = mail->avgC + (mail->tempC - holdC);
        myTemp->avgC = myTemp->avgC + (myTemp->tempC - holdC);
        mail->avgF = mail->avgF + (mail->tempF - holdF);
        myTemp->avgF = myTemp->avgF + (myTemp->tempF - holdF);

        numOfReadings++; //increase total number of readings
        totalC += myTemp->tempC; //increase total temperature read in C
        totalF += myTemp->tempF; //increase total temperature read in F
        
        SemFlag.release(); //signal temperature reading
        
    
}

void TextLCD(void const *args){
    Temperature* myTemp = (Temperature*)args;
    char buffer[20];

    while(cont) {       // loop to update LCD interface
        lcd.cls();
        lcd.locate(0,0);
        lcd.printf("Temperature Logger");
        lcd.locate(0,1);
        lcd.printf("    Time = %.1f", t.read());
        lcd.locate(0,2);
        lcd.printf("C = %.1f F = %.1f", myTemp->tempC, myTemp->tempF);
        lcd.locate(0,3);
        lcd.printf("C/s= %.1f F/s= %.1f", (myTemp->avgC/t.read()), (myTemp->avgF/t.read()));
        
        Thread::wait(50);    // wait 0.5ns to reduce the number of double sent times 
        if (t.read() >= TimeToRun) //break loop if time to end
            cont = false;
    
    }
    snprintf(buffer, sizeof(buffer), ",#,%.1f,%.1f,%.1f, \n", (float)totalF/numOfReadings, (myTemp->avgF/t.read()), t.read()); //puts final data to application in buffer
    client.send_all(buffer, sizeof(buffer));//sends final data to application
    lcd.cls();
    
    /***** Display all final data on LCD*****/
    lcd.locate(0,0);
    lcd.printf("COMPLETE");
    lcd.locate(0,1);
    lcd.printf("AvgC= %.1f", (float)totalC/numOfReadings);
    lcd.locate(0,2);
    lcd.printf("AvgF= %.1f", (float)totalF/numOfReadings);
    lcd.locate(0,3);
    lcd.printf("C/s= %.1f F/s= %.1f", (myTemp->avgC/t.read()), (myTemp->avgF/t.read()));

    /***** Play final noise on speaker******/
    Speaker.period(1.0/500.0); // 500hz period
    Speaker =0.01; //50% duty cycle - max volume
    Thread::wait(1500);    // wait 1.5s
    Speaker=0.0; // turn off audio
    led1 = 1;
    led2 = 1;
    led3 = 1;
    led4 = 1;
    cont = false;

}

 //write temp from analalog port 17 to shared buffer args
 //This is a real-time task and will only execute every given interval from application
void readTemp(void const *args){
        if (pause == true){
            while(1){
                if ((int)t.read()%timeInterval == 0 && pause == false){
                  break;
                }
            }    
        }
        //turn on LED's when reading temp
    
        
        Temperature *mail = mail_box.alloc(); //allocate mailbox to memory
        
        calcTemp(mail, args);
        mail_box.put(mail); //put it on the mailbox
        mail_box.free(mail);  

}

// Speaker/LED thread - will play noise and flash LED everytime temp is read
void speaker(void const *args)
{
    SemFlag.wait();
    while(cont) {
            SemFlag.wait(); //wait for temperature reading
            
            led1 = 1;
            led2 = 1;
            led3 = 1;
            led4 = 1;
            Speaker.period(1.0/1000.0); // 500hz period
            Speaker =0.05; //50% duty cycle - max volume
            Thread::wait(1);    // short wait
            Speaker=0.0; // turn off audio
            Thread::wait(500);    // wait 0.5s
            led1 = 0;
            led2 = 0;
            led3 = 0;
            led4 = 0; 
        
    }
}

void Pause(void const *args) //will await message from application for pausing/resume
{
    
    char buffer[5];
    while(cont) {
        client.receive(buffer, sizeof(buffer)); //will wait until info recieved
        printf("%s\n", buffer);
        if (strcmp(buffer, "Pause") == 0){ //pauses all except for temp reader (stops timer)
            t.stop();
            pause = true;   
        }
        else if (strcmp(buffer, "Resum") == 0){ //resumes all threads
            t.start(); 
            pause = false;
        }
        else if (strcmp(buffer, "Stopn") == 0){ // stops all threads
            cont = false; 
            break;  
        }
        memset(&buffer, 0, sizeof(buffer)); //clear buffer
    }
}

/****** Thread responsible for writing to the USB Flash Drive**********/
void USBWrite(void const *args)
{
    FILE *fp = fopen( "/" FSNAME "/Temperature_Log.txt", "w"); //open file on flash drive
    if ( fp == NULL ) //if error
    {
        error("Could not open file for write\n");
    }
    else
        printf("File opened successfuly\n");
            
    while(cont){
        if (pause == false){
            
            osEvent evt = mail_box.get(); // this will wait until something is put onto the mailbox
            if (evt.status == osEventMail && pause == false && cont == true) { //if there is mail and application is not paused or stopped
                Temperature *mail = (Temperature*)evt.value.p;
                fprintf(fp, "Timestamp(s): |  %.1f Temperature(C) = %.1f  Average Change = %.1f    |    ", mail->timeStamp, mail->tempC, (mail->avgC/t.read()));
                printf("Timestamp: %.1f Temperature(C) = %.1f  Average Change = %.1f    |   ", mail->timeStamp, mail->tempC, (mail->avgC/t.read()));
                fprintf(fp, "%.1f Temperature(F) = %.1f  Average Change = %.1f\r\n", mail->tempF, (mail->avgF/t.read()));
                printf("Temperature(F) = %.1f  Average Change = %.1f\n", mail->tempF, (mail->avgF/t.read()));
                mail_box.free(mail);
            }
            
            
        }
    }
    fclose(fp); //closes file
}

int main() {
    
    Temperature myTemp;
    myTemp.avgF = 0;
    myTemp.avgC = 0;
    myTemp.tempC = ((tempSensor*3.3)-0.600)*100.0;
    myTemp.tempF = (9.0*myTemp.tempC)/5.0 + 32.0;
    
    led1 = 0;
    led2 = 0;
    led3 = 0;
    led4 = 0;
    DIR *d;
    struct dirent *p;
    char* runtime;
    char* interval;
    int n = 0;
    char buffer[15];
    
    d = opendir("/" FSNAME);
    
    /*** Display all files on the flash drive ***/
    printf("\nList of files on the flash drive:\n"); 
    if ( d != NULL )
    {
        while ( (p = readdir(d)) != NULL )
        {
            printf(" - %s\n", p->d_name); 
        }
    }
    else
    {
        error("Could not open directory!");
    }
        
    EthernetInterface eth;
    eth.init(); //Use DHCP
    eth.connect();
    printf("\nIP Address is %s\n", eth.getIPAddress()); //print IP of MBED
    
    lcd.locate(0,0);
    lcd.printf("Wait for connection...");
    
    TCPSocketServer server;
    server.bind(SERVER_PORT);
    server.listen();
    
    printf("\nWait for new connection...\n");
    server.accept(client); //wait and accept new client
    client.set_blocking(true, 1500); // Set to blocking    
    printf("Connection from: %s\n", client.get_address()); //print JAVA IP on COM port
        
    /***** Print Ethernet Info on LCD Interface ****/
    lcd.cls();
    lcd.locate(0,0);
    lcd.printf("My IP: %s", eth.getIPAddress()); 
    lcd.locate(0,1);
    lcd.printf("Wait for info from");
    lcd.locate(0,2);
    lcd.printf("%s", client.get_address());
        
    n = client.receive(buffer, sizeof(buffer)); //will wait until info recieved frin JAVA application for runtime and interval
        if (n == 0) //if error receiving
        printf("Error recieving data");
    runtime = strtok(buffer, " "); //tokenize buffer
    interval = strtok(NULL, " ");    
        
    TimeToRun = atoi(runtime);
    timeInterval = atoi(interval);
        
    /************** Start all threads **********************/
    printf("Runtime %d Interval %d\n", TimeToRun, timeInterval);
    Thread writeToLog(USBWrite);
    Thread speakerThread(speaker);
    Thread pauseThread(Pause);
    lcd.cls();
    Thread LCD_thread(TextLCD, &myTemp);
    RtosTimer tempReader(readTemp, osTimerPeriodic, &myTemp);
    /*******************************************************/
    t.start(); //start timer
    tempReader.start(timeInterval*1000); //setup real time task with period of 5 seconds
    memset(&buffer, 0, sizeof(buffer)); //Preclear buffer
    while(cont){
            if (pause == true){
                tempReader.stop();
                pauseFlag = 0;
            }
            
            if ((int)t.read()%timeInterval == 0 && pauseFlag == 0){
                tempReader.start(timeInterval*1000); //setup real time task with period of 5 seconds
                led1 = 1;
                led2 = 1;
                led3 = 1;
                led4 = 1;
                
                Temperature *mail = mail_box.alloc(); //allocate mailbox to memory
                calcTemp(mail, &myTemp);
                mail_box.put(mail); //put it on the mailbox
                
                pauseFlag = 1;  
            }
            snprintf(buffer, sizeof(buffer), ",%d,%.1f\n", (int)t.read(), myTemp.tempF); // put information in char* buffer for TCP transmit
            printf("Sent through socket: %s\n", buffer); //Verify on COM port
            client.send_all(buffer, sizeof(buffer)); //Send buffer over TCP
            //Thread::wait(500);    // wait 0.5ns to reduce the number of double sent times
                if (n == 0) // If failed
                    printf("Error sending to client");     
            memset(&buffer, 0, sizeof(buffer)); //Clear buffer
            
    }
    tempReader.stop();
    client.close(); //close tcp connection
    Thread::wait(osWaitForever);
    
}
